var Utility = {
	// ts_utility

	// These shouldn't really change, except for locations and filenames
	defaults: {
		configDir: 'config/',
		metaConfigFile: 'meta.json',
		configFile: 'local_test.json',
		xmlParent: 'Object',
		configAttribute: 'config',
		xmlHeader: '<?xml version="1.0" encoding="utf-8"?>\n',
		files: {
			deploy_prod: 'deploy_prod.json',
			deploy_test: 'deploy_test.json',
			local_prod: 'local_prod.json',
			local_test: 'local_test.json',
		}
	},

	formatName: function(name) {
		// Gives the given string proper casing (proper names, etc)
		if (name) {
			return name.replace(/\w\S*/g, function(txt) {
				return txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase();
			});
		}
	},
	
	formatDate: function(string, isBackwards) {
		// isBackwards notes if this date will need to be YYYY-mm-dd (true) instead of mm-dd-YYYY (false)
		// HTML <input type='date'> elements expect isBackwards = false
		// SAP expects isBackwards = true
		if (string === null) return '';
		if (isBackwards === null) isBackwards = false;
		// Default to today's date if no date string is given, or if it's empty
		// Default to today's date if no date string is given, or if it's empty
		if (typeof string === 'undefined' || string === '') {
			console.info('formatDateString() defaulting to today\'s date since none was given');
			var date = new Date().toISOString();
		} else {
			if(typeof string === 'string') {
				string = string.substring(0, 10)				
			}
			var date = new Date(string).toISOString();
		}
	
		if (isBackwards) {
			return date.substr(0, 4) + '-' + date.substr(5, 2) + '-' + date.substr(8, 2)
		} else {
			return date.substr(5, 2) + '/' + date.substr(8, 2) + '/' + date.substr(0, 4);
		}
	},
	
	parseResponse: function(data, isSingle, masterKey) {
		// A helper function to convert an Automat websocket query result to an array of objects
		// where array = [
		//		{column[0]: row[0][0], column[1]: row[0][1], ...}, 
		//		{column[0]: row[1][0], column[1]: row[1][1], ...},  
		//		{column[0]: row[2][0], column[1]: row[2][1], ...},  
		//		...
		// ]
		//
		// If isSingle = true, the expected response is a single row of data that would not benefit from the
		// above architecture. Instead, a single object will be created and returned, using column names as
		// parameters and row values as the values.
		// 
		// The main difference is that isSingle creates the object with the importance given to the columns,
		// whereas !isSingle gives importance to the rows
		
		if (typeof data == 'undefined' || typeof data !== 'object' || data == '') {
			console.error('parseResponse() missing or bad parameter 1: data; this should be a query response as it comes from the websocket');
			return false;
		}
		
		if (typeof isSingle == 'undefined' || isSingle === '') {
			// Since it would be better to pass back all data instead of just one row's worth, we'll default to !isSingle
			isSingle = false;
			// console.info('parseResponse() missing parameter 2: bool isSingle; defaulting to array of objects (false)');
		}
		
		var columns = data.Data.Result.Result.Columns;
		var rows = data.Data.Result.Result.Rows;
		var result = null;
		
		if (rows.length === 1 && columns.length === rows[0].length) {
			// console.warn('parseResponse() has noticed that only 1 row has been returned. Consider setting isSingle to true if this is always the case');
			// console.log(data);
		}
		
		if (isSingle === true && rows.length > 1) {
			console.warn(
				'parseResponse() has been set to single mode (isSingle = true), but is parsing multiple rows of data. ' +
				'You will only get the first row returned in this configuration. '+
				'Consider setting isSingle to false if this is always the case'
			);
			console.log(data);
		}

		// All of that error handling for a few quick loops
		if (isSingle) {
			result = {};
			for (var i = 0; i < columns.length; i++) {
				var type = columns[i].Type;
				result[columns[i].Name] = (type.match(/int/i)) ? parseInt(rows[i]) : rows[i];
			}
		} else {
			result = [];
			for (var i = 0; i < rows.length; i++) {
				result.push({});
				for (var j = 0; j < columns.length; j++) {
					var name = columns[j]["Name"];
					var type = columns[j]["Type"];
					var val = rows[i][j];
					//If the value is an integer (server lets us know) then parse it into one
					if (type.match(/int/i)) {	result[i][name] = parseInt(val);	} 
					else {					result[i][name] = val;			}
				}
			}
		}
		if (masterKey) {
			var parsedObject = {}
			if(typeof masterKey === 'string') {
				result.forEach(function(object) {
					parsedObject[object[masterKey]] = object
				})
			} else if(typeof masterKey === 'object') {
				var objToReturn = Object.keys(masterKey);
				objToReturn.forEach(function(obj) {
					var key = masterKey[obj]['column'];
					if(!masterKey[obj].hasOwnProperty('type') || masterKey[obj]['type'] === 'object') {
						parsedObject[obj] = {};
						result.forEach(function(object) {
							parsedObject[obj][object[key]] = object
						});
					} else if(masterKey[obj]['type'] === 'arrayByColumn') {
						parsedObject[obj] = {};
						result.forEach(function(object) {
							if(!parsedObject[obj].hasOwnProperty(object[key])) parsedObject[obj][object[key]] = [];
							parsedObject[obj][object[key]].push(object);
						});
					}
				})
			}
			result = parsedObject
		}			
		return result;
	},
	
	htmlSpecialChars: function(unsafe) {
		// Sanitizes a string for use in XML 
		if(unsafe === null) unsafe = '';
		if (typeof unsafe === 'string') return unsafe.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;");
		return unsafe;
	},
	
	objectToXML: function(object, root) {
		// http://goessner.net/download/prj/jsonxml/
		// ... but I made it readable and just better overall
		var toXml = function(value, key, depth) {
			var xml = '';
			// Array elements need to repeat the parent element as a wrapper for each item in the array
			if (value instanceof Array) {
				for (var i = 0; i < value.length; i++) {
					xml += depth + toXml(value[i], key, depth+'\t') + '\n';
				}
				
			} else if (typeof value === 'object') {
				var hasChild = false;
				xml += depth + '<' + key;
				for (var element in value) {
					// Handle attributes, e.g. <book id=''> === book: { '@id': ''  }
					if (element.charAt(0) === '@') {
						xml += ' ' + element.substr(1) + '="' + Utility.htmlSpecialChars(value[element].toString()) + '"';
					} else {
						hasChild = true;
					}
				}
				xml += (hasChild) ? '>\n' : '/>\n';
				if (hasChild) {
					for (var element in value) {
						if (element === '#text') {
							xml += Utility.htmlSpecialChars(value[element]);
						} else if (element === '#cdata') {
							xml += '<![CDATA[' + value[element] + ']]>';
						} else if (element.charAt(0) != '@'){
							xml += toXml(value[element], element, depth+"\t");
						}
					}
					
					var lastChar = xml.charAt(xml.length-1);
					xml += ((lastChar === '\n') ? depth : '') + '</' + key + '>\n';
				}
			} else {
				xml += depth + '<' + key + '>' + Utility.htmlSpecialChars(value.toString()) +  '</' + key + '>\n';
			}
			
			return xml;
		}
		
		var xml = '<?xml version="1.0" encoding="utf-8"?>\n<' + root + '>\n';
		for (var element in object) {
			//Automat throws a fit when an element is null
			if (object[element] === null) object[element] = '';
			if (object.hasOwnProperty(element)) xml += toXml(object[element], element, '\t');
		}
		return xml + '</' + root + '>';
	},

	isBadString: function(string) {
		// Returns true if the given string is empty, undefined, or not a string at all; otherwise, returns false
		var whitespaceReg = /^[\s]*$/;

		return (
		string === null
		|| typeof string === "undefined"
		|| (typeof string === "string" && whitespaceReg.test(string))
		|| (typeof string === "Object" && string.constructor.name === "String" && whitespaceReg.test(string.toString()))
		);
	},
	
	makeRandomString: function (length, bits) {
		// Creates a randomized string of the given length, in the given bits
		// Useful for creating generic but unique IDs, session keys, or whatever
		length = length || 20;
		bits = bits || 36;
		var result = "";
		var temp;
		while (result.length < length) {
			temp = Math.random().toString(bits).slice(2);
			result += temp.slice(0, Math.min(temp.length, (length - result.length)));
		}
		return result.toUpperCase();
	},
		
	getBasename: function(path) {
		// Returns the filename component of a path
		return path.split(/[\\/]/).pop();	
	},
	
	castValue: function(value, type) {
		// Casts a value in a format specified. While this is mostly just stock parse___() functions,
		// it also allows us to convert to specialized formats that would require their own function,
		// but that don't really deserve their own spot in the global.
		switch (type) {
			case 'float': {
				inputVal = parseFloat(inputVal);
				break;
			}
			case 'int': {
				inputVal = parseInt(inputVal);
				break;
			}
		}
		return inputVal;
	},
	
	flattenObject: function(object) {
		// https://gist.github.com/penguinboy/762197
		// Flattens objects into a single-level object with keys that are . separated strings
		// of the previous object's keys, and the values that lived at the bottom of those paths. 
		var result = {};
		for (var item in object) {
			if (object.hasOwnProperty(item)) {
				if (object[item] instanceof Object) {
					var subObj = this.flattenObject(object[item]);
					for (var subItem in subObj) {
						if (subObj.hasOwnProperty(subItem)) {
							result[item + '.' + subItem] = subObj[subItem];
						}
					}
				} else {
					result[item] = object[item];
				}
			}
		}
		this.currentFile.flat = result;
		return result;
	},
	
	getKeyPath: function(target, path, asArray) {
		// http://blog.wax-o.com/2014/01/how-to-find-deep-and-get-parent-in-javascript-nested-objects-with-recursive-functions-and-the-reference-concept-level-beginner/
		// Returns a flat . separated string or array that shows the path from a property at
		// any level of an object up to the top level
		if (asArray === '' || asArray === 'undefined') asArray = false;
		
		if (target.parent) {
			path = target.parent.name + '.' + path;
			return this.getKeyPath(target.parent, path);
		} else {
			return (asArray) ? path.split('.') : path;
		}
	},
	
	storage: {
		setObject: function(objectKey, object) {
			localStorage.setItem(objectKey, JSON.stringify(object));
		},
		
		getObject: function(objectKey) {
			var retrievedObject = JSON.parse(localStorage.getItem(objectKey))
			return retrievedObject
		}, 
		
		appendToObject: function(objectKey, objectToAppend) {
			var retrievedObject = JSON.parse(localStorage.getItem(objectKey))
			Object.keys(objectToAppend).map(function(key) {
				retrievedObject[key] = objectToAppend[key]
			})
			localStorage.setItem(objectKey, JSON.stringify(retrievedObject));
		},
		
		deleteObject: function(objectKey) {
			localStorage.setItem(objectKey, '');
		},
		
		//Timestamp function that can get, set, or find the difference between time stamps.
		getTimeStamps: function() {
			var timeStamps = {
				previousTimeStamp: localStorage.getItem('previousTimeStamp'),
				currentTimeStamp: localStorage.getItem('currentTimeStamp')
			}
			return timeStamps;
		},
		setTimeStamps: function() {
			localStorage.setItem('previousTimeStamp',  localStorage.getItem('currentTimeStamp'))
			localStorage.setItem('currentTimeStamp',  Date.now())
		},
		getTimeStampDifference: function() {
			return (localStorage.getItem('currentTimeStamp') - localStorage.getItem('previousTimeStamp'));
		}
	},
	
	b64ToBlob: function(b64Data, contentType, sliceSize) {
		//This was stolen from stackoverflow, it shifts a b64 string into a blob, blobs are used
		//to save files in the front end. At this point in time it doesn't matter that much what it
		//exactly is doing because it just shifts two datatypes. 
		contentType = contentType || '';
		sliceSize = sliceSize || 512;

		var byteCharacters = atob(b64Data);
		var byteArrays = [];

		for (var offset = 0; offset < byteCharacters.length; offset += sliceSize) {
			var slice = byteCharacters.slice(offset, offset + sliceSize);

			var byteNumbers = new Array(slice.length);
			for (var i = 0; i < slice.length; i++) {
				byteNumbers[i] = slice.charCodeAt(i);
			}

			var byteArray = new Uint8Array(byteNumbers);

			byteArrays.push(byteArray);
		}

		var blob = new Blob(byteArrays, {type: contentType});
		return blob;
	}
}